// prisma/seed.cjs
/* eslint-disable @typescript-eslint/no-var-requires */
const { PrismaClient } = require("@prisma/client");
const bcrypt = require("bcryptjs"); // <-- use bcryptjs

const prisma = new PrismaClient();

async function main() {
  console.log("🌱 Starting database seed...");

  // Create hospital-specific roles
  const roles = await Promise.all([
    prisma.role.upsert({
      where: { name: "Medical Equipment Technician" },
      update: {},
      create: {
        name: "Medical Equipment Technician",
        description:
          "Specializes in repairing and maintaining medical devices like ventilators, monitors, and diagnostic equipment",
        isActive: true,
      },
    }),
    prisma.role.upsert({
      where: { name: "Radiology Equipment Specialist" },
      update: {},
      create: {
        name: "Radiology Equipment Specialist",
        description:
          "Expert in MRI, CT, X-ray machines and imaging systems maintenance",
        isActive: true,
      },
    }),
    prisma.role.upsert({
      where: { name: "Laboratory Systems Technician" },
      update: {},
      create: {
        name: "Laboratory Systems Technician",
        description:
          "Maintains laboratory equipment, analyzers, and testing systems",
        isActive: true,
      },
    }),
    prisma.role.upsert({
      where: { name: "Hospital IT Specialist" },
      update: {},
      create: {
        name: "Hospital IT Specialist",
        description:
          "Manages EMR systems, network infrastructure, and hospital software",
        isActive: true,
      },
    }),
    prisma.role.upsert({
      where: { name: "Emergency Equipment Technician" },
      update: {},
      create: {
        name: "Emergency Equipment Technician",
        description:
          "Specializes in emergency department equipment like defibrillators and life support systems",
        isActive: true,
      },
    }),
  ]);

  console.log("✅ Created roles:", roles.map((r) => r.name).join(", "));

  // Create users
  const users = [
    {
      email: "admin@hospital.local",
      password: "admin12345",
      name: "System Administrator",
      role: "ADMIN",
    },
    {
      email: "tech1@hospital.local",
      password: "tech12345",
      name: "Dr. Sarah Martinez",
      role: "TECHNICIAN",
      customRoleId: roles[0].id, // Medical Equipment Technician
      ethAddress: "0x742d35Cc6634C0532925a3b8D0C9e2C0C0C0C0C0",
      hourlyRate: 45.0,
    },
    {
      email: "tech2@hospital.local",
      password: "tech12345",
      name: "James Chen",
      role: "TECHNICIAN",
      customRoleId: roles[1].id, // Radiology Equipment Specialist
      ethAddress: "0x8ba1f109551bD432803012645ac136c0C0C0C0C0",
      hourlyRate: 50.0,
    },
    {
      email: "tech3@hospital.local",
      password: "tech12345",
      name: "Maria Rodriguez",
      role: "TECHNICIAN",
      customRoleId: roles[2].id, // Laboratory Systems Technician
      ethAddress: "0x9ba1f109551bD432803012645ac136c0C0C0C0C0",
      hourlyRate: 42.0,
    },
    {
      email: "user1@hospital.local",
      password: "user12345",
      name: "Dr. Emily Thompson",
      role: "USER",
    },
    {
      email: "user2@hospital.local",
      password: "user12345",
      name: "Nurse Michael Johnson",
      role: "USER",
    },
  ];

  for (const userData of users) {
    const { password, customRoleId, ethAddress, hourlyRate, ...userInfo } =
      userData;
    const passwordHash = await bcrypt.hash(password, 10);

    const existing = await prisma.user.findUnique({
      where: { email: userData.email },
    });
    if (existing) {
      console.log(`⚠️  User already exists: ${userData.email}`);
      continue;
    }

    await prisma.user.create({
      data: {
        ...userInfo,
        passwordHash,
        customRoleId: customRoleId || undefined,
        ethAddress: ethAddress || undefined,
        hourlyRate: hourlyRate || undefined,
      },
    });

    console.log(
      `✅ Created ${userData.role}: ${userData.email} (${userData.name})`
    );
  }

  // Create hospital departments
  const departments = await Promise.all([
    prisma.department.upsert({
      where: { name: "Emergency Department" },
      update: {},
      create: {
        name: "Emergency Department",
        isVisible: true,
      },
    }),
    prisma.department.upsert({
      where: { name: "Intensive Care Unit" },
      update: {},
      create: {
        name: "Intensive Care Unit",
        isVisible: true,
      },
    }),
    prisma.department.upsert({
      where: { name: "Radiology Department" },
      update: {},
      create: {
        name: "Radiology Department",
        isVisible: true,
      },
    }),
    prisma.department.upsert({
      where: { name: "Laboratory Services" },
      update: {},
      create: {
        name: "Laboratory Services",
        isVisible: true,
      },
    }),
    prisma.department.upsert({
      where: { name: "Hospital IT Systems" },
      update: {},
      create: {
        name: "Hospital IT Systems",
        isVisible: true,
      },
    }),
  ]);

  console.log(
    "✅ Created departments:",
    departments.map((d) => d.name).join(", ")
  );

  // Create hospital elements/equipment
  const elements = await Promise.all([
    // Emergency Department elements
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[0].id, // Emergency Department
          name: "Defibrillator",
        },
      },
      update: {},
      create: {
        name: "Defibrillator",
        departmentId: departments[0].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[0].id, // Emergency Department
          name: "Ventilator",
        },
      },
      update: {},
      create: {
        name: "Ventilator",
        departmentId: departments[0].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[0].id, // Emergency Department
          name: "Patient Monitor",
        },
      },
      update: {},
      create: {
        name: "Patient Monitor",
        departmentId: departments[0].id,
        isVisible: true,
      },
    }),

    // ICU elements
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[1].id, // ICU
          name: "ICU Bed",
        },
      },
      update: {},
      create: {
        name: "ICU Bed",
        departmentId: departments[1].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[1].id, // ICU
          name: "Infusion Pump",
        },
      },
      update: {},
      create: {
        name: "Infusion Pump",
        departmentId: departments[1].id,
        isVisible: true,
      },
    }),

    // Radiology elements
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[2].id, // Radiology
          name: "MRI Scanner",
        },
      },
      update: {},
      create: {
        name: "MRI Scanner",
        departmentId: departments[2].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[2].id, // Radiology
          name: "CT Scanner",
        },
      },
      update: {},
      create: {
        name: "CT Scanner",
        departmentId: departments[2].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[2].id, // Radiology
          name: "X-Ray Machine",
        },
      },
      update: {},
      create: {
        name: "X-Ray Machine",
        departmentId: departments[2].id,
        isVisible: true,
      },
    }),

    // Laboratory elements
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[3].id, // Laboratory
          name: "Blood Analyzer",
        },
      },
      update: {},
      create: {
        name: "Blood Analyzer",
        departmentId: departments[3].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[3].id, // Laboratory
          name: "Microscope",
        },
      },
      update: {},
      create: {
        name: "Microscope",
        departmentId: departments[3].id,
        isVisible: true,
      },
    }),

    // IT Systems elements
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[4].id, // Hospital IT
          name: "EMR System",
        },
      },
      update: {},
      create: {
        name: "EMR System",
        departmentId: departments[4].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[4].id, // Hospital IT
          name: "Network Server",
        },
      },
      update: {},
      create: {
        name: "Network Server",
        departmentId: departments[4].id,
        isVisible: true,
      },
    }),
    prisma.element.upsert({
      where: {
        departmentId_name: {
          departmentId: departments[4].id, // Hospital IT
          name: "Workstation Computer",
        },
      },
      update: {},
      create: {
        name: "Workstation Computer",
        departmentId: departments[4].id,
        isVisible: true,
      },
    }),
  ]);

  console.log("✅ Created elements:", elements.map((e) => e.name).join(", "));

  // Create role-element assignments
  console.log("\n🔗 Creating role-element assignments...");

  // Medical Equipment Technician - Emergency Department equipment
  try {
    await prisma.roleElement.createMany({
      data: [
        {
          roleId: roles[0].id, // Medical Equipment Technician
          elementId: elements[0].id, // Defibrillator
        },
        {
          roleId: roles[0].id,
          elementId: elements[1].id, // Ventilator
        },
        {
          roleId: roles[0].id,
          elementId: elements[2].id, // Patient Monitor
        },
      ],
    });
  } catch (error) {
    if (error.code !== "P2002") {
      // Not a unique constraint error
      throw error;
    }
    console.log(
      "⚠️  Role-element assignments already exist for Medical Equipment Technician"
    );
  }

  // Radiology Equipment Specialist - Radiology equipment
  try {
    await prisma.roleElement.createMany({
      data: [
        {
          roleId: roles[1].id, // Radiology Equipment Specialist
          elementId: elements[3].id, // MRI Scanner
        },
        {
          roleId: roles[1].id,
          elementId: elements[4].id, // CT Scanner
        },
        {
          roleId: roles[1].id,
          elementId: elements[5].id, // X-Ray Machine
        },
      ],
    });
  } catch (error) {
    if (error.code !== "P2002") {
      throw error;
    }
    console.log(
      "⚠️  Role-element assignments already exist for Radiology Equipment Specialist"
    );
  }

  // Laboratory Systems Technician - Laboratory equipment
  try {
    await prisma.roleElement.createMany({
      data: [
        {
          roleId: roles[2].id, // Laboratory Systems Technician
          elementId: elements[6].id, // Blood Analyzer
        },
        {
          roleId: roles[2].id,
          elementId: elements[7].id, // Microscope
        },
        {
          roleId: roles[2].id,
          elementId: elements[8].id, // Centrifuge
        },
      ],
    });
  } catch (error) {
    if (error.code !== "P2002") {
      throw error;
    }
    console.log(
      "⚠️  Role-element assignments already exist for Laboratory Systems Technician"
    );
  }

  // Hospital IT Specialist - IT equipment
  try {
    await prisma.roleElement.createMany({
      data: [
        {
          roleId: roles[3].id, // Hospital IT Specialist
          elementId: elements[9].id, // EMR System
        },
        {
          roleId: roles[3].id,
          elementId: elements[10].id, // Network Server
        },
        {
          roleId: roles[3].id,
          elementId: elements[11].id, // Workstation Computer
        },
      ],
    });
  } catch (error) {
    if (error.code !== "P2002") {
      throw error;
    }
    console.log(
      "⚠️  Role-element assignments already exist for Hospital IT Specialist"
    );
  }

  // Emergency Equipment Technician - Emergency and ICU equipment
  try {
    await prisma.roleElement.createMany({
      data: [
        {
          roleId: roles[4].id, // Emergency Equipment Technician
          elementId: elements[0].id, // Defibrillator
        },
        {
          roleId: roles[4].id,
          elementId: elements[1].id, // Ventilator
        },
        {
          roleId: roles[4].id,
          elementId: elements[2].id, // Patient Monitor
        },
      ],
    });
  } catch (error) {
    if (error.code !== "P2002") {
      throw error;
    }
    console.log(
      "⚠️  Role-element assignments already exist for Emergency Equipment Technician"
    );
  }

  console.log("✅ Created role-element assignments");

  console.log("\n🎉 Database seeded successfully!");
  console.log("\n📋 Login Credentials:");
  console.log("Admin: admin@hospital.local / admin12345");
  console.log(
    "Technicians: tech1@hospital.local, tech2@hospital.local, tech3@hospital.local / tech12345"
  );
  console.log("Users: user1@hospital.local, user2@hospital.local / user12345");
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
